<?php
/*--------------------------------------------------------------
   GambioOrdersList.php 2021-08-18
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2021 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
 -------------------------------------------------------------*/

declare(strict_types=1);

namespace Gambio\Admin\Modules\Statistics\App\Overview\Model\Entities\WidgetDefinition;

use DateTime;
use Doctrine\DBAL\Connection;
use Gambio\Admin\Modules\Statistics\App\Data\Factory as DataFactory;
use Gambio\Admin\Modules\Statistics\App\Overview\Factory as OverviewFactory;
use Gambio\Admin\Modules\Statistics\App\Overview\Factory\Option\Predefined\MaxEntriesDropdown;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Collections\Options;
use Gambio\Admin\Modules\Statistics\Model\ValueObjects\Data;
use Gambio\Core\Application\ValueObjects\UserPreferences;
use NumberFormatter;

class GambioOrdersList extends GambioWidgetDefinition
{
    private const TYPE = 'gambio.orders.list';
    
    private const VERSION = '0.1.0';
    
    private const ICON = "data:image/png;base64,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";
    
    private const WIDGET_TITLE = [
        self::LANGUAGE_CODE_GERMAN  => 'Bestellungen (Liste)',
        self::LANGUAGE_CODE_ENGLISH => 'Orders (List)'
    ];
    
    private const TABLE_DATE_COLUMN_DATE_INPUT_FORMAT = 'Y-m-d H:i:s';
    
    private const TABLE_DATE_COLUMN_DATE_OUTPUT_FORMAT = 'd.m.Y H:i';
    
    private const TABLE_COLUMNS = [
        'customer' => [
            self::LANGUAGE_CODE_GERMAN  => 'Kunde',
            self::LANGUAGE_CODE_ENGLISH => 'Customer',
        ],
        'date'     => [
            self::LANGUAGE_CODE_GERMAN  => 'Bestelldatum',
            self::LANGUAGE_CODE_ENGLISH => 'Order date',
        ],
        'amount'   => [
            self::LANGUAGE_CODE_GERMAN  => 'Summe',
            self::LANGUAGE_CODE_ENGLISH => 'Amount',
        ],
        'status'   => [
            self::LANGUAGE_CODE_GERMAN  => 'Status',
            self::LANGUAGE_CODE_ENGLISH => 'Status',
        ],
    ];
    
    /**
     * @var DataFactory
     */
    private $dataFactory;
    
    /**
     * @var UserPreferences
     */
    private $userPreferences;
    
    /**
     * @var Connection
     */
    private $connection;
    
    /**
     * @var NumberFormatter
     */
    private $numberFormatter;
    
    
    /**
     * @inheritDoc
     */
    public function __construct(
        OverviewFactory $overviewFactory,
        DataFactory $dataFactory,
        UserPreferences $userPreferences,
        Connection $connection,
        NumberFormatter $numberFormatter
    ) {
        $this->dataFactory     = $dataFactory;
        $this->connection      = $connection;
        $this->userPreferences = $userPreferences;
        $this->numberFormatter = $numberFormatter;
        
        parent::__construct($overviewFactory->createType(self::TYPE),
                            $overviewFactory->createVersion(self::VERSION),
                            $overviewFactory->createIconUsingData(self::ICON),
                            $overviewFactory->useVisualizations()->createTable(),
                            $overviewFactory->useOptions()->createOptions($overviewFactory->useOptions()
                                                                              ->usePredefined()
                                                                              ->createMaxEntriesDropdown($overviewFactory)),
                            $overviewFactory->createTitles($overviewFactory->createTitle($overviewFactory->createLanguageCode(self::LANGUAGE_CODE_GERMAN),
                                                                                         self::WIDGET_TITLE[self::LANGUAGE_CODE_GERMAN]),
                                                           $overviewFactory->createTitle($overviewFactory->createLanguageCode(self::LANGUAGE_CODE_ENGLISH),
                                                                                         self::WIDGET_TITLE[self::LANGUAGE_CODE_ENGLISH])));
    }
    
    
    /**
     * @inheritDoc
     */
    public function data(Options $options): Data
    {
        $languageCode = $this->userPreferences->languageId()
                        === self::LANGUAGE_ID_GERMAN ? self::LANGUAGE_CODE_GERMAN : self::LANGUAGE_CODE_ENGLISH;
        
        $orders = $this->connection->createQueryBuilder()
            ->select([
                         'orders.customers_name AS customer',
                         'orders.date_purchased AS date',
                         'orders_total.value AS amount',
                         'orders.currency AS currency',
                         'orders_status.orders_status_name AS status'
                     ])
            ->from('orders')
            ->join('orders',
                   'orders_total',
                   'orders_total',
                   'orders.orders_id = orders_total.orders_id')
            ->join('orders',
                   'orders_status',
                   'orders_status',
                   'orders.orders_status = orders_status.orders_status_id')
            ->where('orders_total.class="ot_total"')
            ->andWhere('orders_status.language_id = :languageId')
            ->orderBy('orders.date_purchased', 'DESC')
            ->setMaxResults((int)$options->getById(MaxEntriesDropdown::ID)->value())
            ->setParameter('languageId', $this->userPreferences->languageId())
            ->execute()
            ->fetchAll();
        
        return $this->dataFactory->useTableData()->createTableData($this->dataFactory->useTableData()
                                                                       ->createColumns($this->dataFactory->useTableData()
                                                                                           ->createTextColumn(self::TABLE_COLUMNS['customer'][$languageCode],
                                                                                                              'customer'),
                                                                                       $this->dataFactory->useTableData()
                                                                                           ->createTextColumn(self::TABLE_COLUMNS['amount'][$languageCode],
                                                                                                              'amount'),
                                                                                       $this->dataFactory->useTableData()
                                                                                           ->createDateColumn(self::TABLE_COLUMNS['date'][$languageCode],
                                                                                                              'date',
                                                                                                              self::TABLE_DATE_COLUMN_DATE_INPUT_FORMAT,
                                                                                                              self::TABLE_DATE_COLUMN_DATE_OUTPUT_FORMAT),
                                                                                       $this->dataFactory->useTableData()
                                                                                           ->createTextColumn(self::TABLE_COLUMNS['status'][$languageCode],
                                                                                                              'status')),
                                                                   $this->dataFactory->useTableData()->createRows(...
                                                                       array_map(function (array $order) {
                                                                           return $this->dataFactory->useTableData()
                                                                               ->createRow($this->dataFactory->useTableData()
                                                                                               ->createRowFields($this->dataFactory->useTableData()
                                                                                                                     ->createRowTextField('customer',
                                                                                                                                          $order['customer']),
                                                                                                                 $this->dataFactory->useTableData()
                                                                                                                     ->createRowTextField('date',
                                                                                                                                          DateTime::createFromFormat(self::TABLE_DATE_COLUMN_DATE_INPUT_FORMAT,
                                                                                                                                                                     $order['date'])
                                                                                                                                              ->format(self::TABLE_DATE_COLUMN_DATE_OUTPUT_FORMAT)),
                                                                                                                 $this->dataFactory->useTableData()
                                                                                                                     ->createRowTextField('amount',
                                                                                                                                          $this->numberFormatter->formatCurrency((float)$order['amount'],
                                                                                                                                                                                 $order['currency'])),
                                                                                                                 $this->dataFactory->useTableData()
                                                                                                                     ->createRowTextField('status',
                                                                                                                                          $order['status'])));
                                                                       },
                                                                           $orders)));
    }
}
